// ============================================================================
//
// Copyright (c) 2010-2016, niceideas.ch - Jerome Kehrli
//
// You may distribute this code under the terms of the GNU LGPL license
// (http://www.gnu.org/licenses/lgpl.html). [^]
//
// ============================================================================



#include "worm.h"

extern Board* board;
extern Engine* engine;
extern Worm* worm;

// class of the worm
// ----------------------------------------------------------------------------

// constructor : fix initial position
Worm::Worm(int x, int y, direction_t dir)
{
	offset = 0;
	bodyPartCount = 0;
	speedCoeffWorm = 1;
    this->body = new Body_Part[BOARD_SIZE * BOARD_SIZE];
    this->initX = x;
    this->initY = y;
    this->initDirection = dir;
    running = true;
}

void Worm::run(void)
{
    running = true;
}

void Worm::pause(void)
{
    running = false;
}

void Worm::switchPause(void)
{
    running = !running;
}

// (re) initialize the worm
void Worm::init(void)
{
    bodyPartCount = 0;
    offset = 0;
    speedCoeffWorm = 10;

    head.setPosition(initX, initY);
    head.setDirection(initDirection);
}

// specifiy the worm speed coefficient
void Worm::fixeCoeff(float coeff)
{
    speedCoeffWorm = coeff;
}

// destructor
Worm::~Worm()
{
    delete[] this->body;
}

// move the worm one unit
void Worm::move(void)
{
    if (running && (offset < OFFSET_RANGE))
    {
        offset += speedCoeffWorm;
    }
}

// how many parts are now in the worm body ?
int Worm::getBodyPartCount(void)
{
    return bodyPartCount;
}

// add a bodypart to the worm
void Worm::addBodyPart(int headX, int headY)
{
    int posX, posY;

    if (bodyPartCount > 0)
    {
        body[bodyPartCount - 1].getPosition(posX, posY);
    }
    else
    {
        posX = headX;
        posY = headY;
    }

    bodyPartCount++;
    body[bodyPartCount - 1].setOrigin(posX, posY, bodyPartCount);

    body[bodyPartCount - 1].setDirection(board->getCell(posX, posY)->getDirection());

    board->getCell(posX, posY)->setUsed();
}

void Worm::moveHead(int nextPositionX, int nextPositionY, bool &quit, int headX, int headY)
{
    head.setPosition(nextPositionX, nextPositionY);
    if (!board->getCell(nextPositionX, nextPositionY)->isFree())
    {
        quit = true;
    }
    else
    {
        board->getCell(headX, headY)->setFree();
        board->getCell(nextPositionX, nextPositionY)->setUsed();
    }
}

// save the new directions
bool Worm::moveCell(void)
{
    int headX, headY;
    int newHeadX, newHeadY;
    int x, y;
    int limit;
    bool doCreateSeed;

    doCreateSeed = false;

    if (offset >= OFFSET_RANGE)
    {
        // ANALYZE HEAD

        // position of the head before the cell switch
        head.getPosition(headX, headY);

        // affect to that position the current head direction
        board->getCell(headX, headY)->setDirection(head.getDirection());

        // move the head one complete cell and mark this cell as busy

        // TODO This can be refactored
        bool quit = false;
        switch (head.getDirection())
        {
        case DIR_RIGHT:
            if (headX + 1 > board->getSizeX() - 1)
                quit = true;
            else
            {
                moveHead(headX + 1, headY, quit, headX, headY);
            }
            break;

        case DIR_UP:
            if (headY + 1 > board->getSizeY() - 1)
                quit = true;
            else
            {
                moveHead(headX, headY + 1, quit, headX, headY);
            }
            break;

        case DIR_LEFT:
            if (headX - 1 < 0)
                quit = true;
            else
            {
                moveHead(headX - 1, headY, quit, headX, headY);
            }
            break;

        case DIR_DOWN:
            if (headY - 1 < 0)
                quit = true;
            else
            {
                moveHead(headX, headY - 1, quit, headX, headY);
            }
            break;
        }
        if (quit)
        {
            return false;
        }

        // Then of that new cell has the seed, we reached it ü
        // so we'll have to re-create a seed and add a body part to the worm

        limit = bodyPartCount;
        // (limt = body_part_number is required to not make the new part move
        // to the next cell along with the worm)

        head.getPosition(newHeadX, newHeadY);
        if (board->matchesSeedCell(newHeadX, newHeadY))
        {
            this->addBodyPart(headX, headY);
            doCreateSeed = true;
        }

        // And now the body parts

        // TODO this can be refactored
        for (int i = 0; i < limit; i++)
        {
            body[i].getPosition(x, y);
            switch (body[i].getDirection())
            {
            case DIR_RIGHT:
                body[i].setPosition(x + 1, y);
                board->getCell(x, y)->setFree();
                board->getCell(x + 1, y)->setUsed();
                break;

            case DIR_UP:
                body[i].setPosition(x, y + 1);
                board->getCell(x, y)->setFree();
                board->getCell(x, y + 1)->setUsed();
                break;

            case DIR_LEFT:
                body[i].setPosition(x - 1, y);
                board->getCell(x, y)->setFree();
                board->getCell(x - 1, y)->setUsed();
                break;

            case DIR_DOWN:
                body[i].setPosition(x, y - 1);
                board->getCell(x, y)->setFree();
                board->getCell(x, y - 1)->setUsed();
                break;
            }

            // set new direction
            body[i].getPosition(x, y);
            body[i].setDirection(board->getCell(x, y)->getDirection());
        }

        // reset offset
        offset = 0;

        if (doCreateSeed)
        {
            board->createRandomSeed();
            doCreateSeed = false;
        }

        // get next direction from the engine
        worm->setDirection(engine->getNext());
    }

    return true;
}

// used to know current offset
int Worm::getOffset(void)
{
    return offset;
}

// get the position of the head
void Worm::getHeadPosition(int &head_x, int &head_y)
{
    head.getPosition(head_x, head_y);
}

// apply the new direction to the worm (i.e. the head)
void Worm::setDirection(direction_t dir)
{
    head.setDirection(dir);
}

// get the direction of the worm (i.e. the head)
direction_t Worm::getDirection(void)
{
    return head.getDirection();
}

int Worm::getNearBodyPartCount(int posX, int posY)
{
    int retCount = 0;

    if (posX > 0)
    {
        Cell* cellUp = board->getCell(posX - 1, posY);
        if (!cellUp->isFree())
        {
            retCount++;
        }
    }

    if (posX < board->getSizeX() - 1)
    {
        Cell* cellDown = board->getCell(posX + 1, posY);
        if (!cellDown->isFree())
        {
            retCount++;
        }
    }

    if (posY > 0)
    {
        Cell* cellLeft = board->getCell(posX, posY - 1);
        if (!cellLeft->isFree())
        {
            retCount++;
        }
    }

    if (posY < board->getSizeY() - 1)
    {
        Cell* cellRight = board->getCell(posX, posY + 1);
        if (!cellRight->isFree())
        {
            retCount++;
        }
    }

    return retCount;
}

void Worm::display(void)
{
    int x, y;

    head.getPosition(x, y);

    glPushMatrix();
        myGlTranslate(f2vt(100.0 * x), f2vt(100.0 * y), 0);

        // TODO factorize with the following
        switch (head.getDirection())
        {
        case DIR_RIGHT:
            myGlTranslate(f2vt(10.0 * ((float) offset / (float) (OFFSET_RANGE / 10))), 0, 0);
            break;
        case DIR_UP:
            myGlTranslate(0, f2vt(10.0 * ((float) offset / (float) (OFFSET_RANGE / 10))), 0);
            break;
        case DIR_LEFT:
            myGlTranslate(f2vt(10.0 * ((float) -offset / (float) (OFFSET_RANGE / 10))), 0, 0);
            break;
        case DIR_DOWN:
            myGlTranslate(0, f2vt(10.0 * ((float) -offset / (float) (OFFSET_RANGE / 10))), 0);
            break;
        }

        // TODO Optimization
        //glCallList (head.display_list);

        head.display();

    glPopMatrix();

    for (int i = 0; i < bodyPartCount; i++)
    {

        body[i].getPosition(x, y);

        glPushMatrix();

            myGlTranslate(f2vt(100.0 * x), f2vt(100.0 * y), 0);

            switch (body[i].getDirection()) {
            case DIR_RIGHT:
                myGlTranslate(f2vt(10.0 * ((float) offset / (float) (OFFSET_RANGE / 10))), 0, 0);
                break;
            case DIR_UP:
                myGlTranslate(0, f2vt(10.0 * ((float) offset / (float) (OFFSET_RANGE / 10))), 0);
                break;
            case DIR_LEFT:
                myGlTranslate(f2vt(10.0 * ((float) -offset / (float) (OFFSET_RANGE / 10))), 0, 0);
                break;
            case DIR_DOWN:
                myGlTranslate(0, f2vt(10.0 * ((float) -offset / (float) (OFFSET_RANGE / 10))), 0);
                break;
            }

            // TODO Optimization
            //glCallList (body[i].display_list);

            body[i].display();

        glPopMatrix();
    }
}
