// ============================================================================
//
// Copyright (c) 2010-2016, niceideas.ch - Jerome Kehrli
//
// You may distribute this code under the terms of the GNU LGPL license
// (http://www.gnu.org/licenses/lgpl.html). [^]
//
// ============================================================================



#include "cell.h"
#include "board.h"
#include "worm.h"


extern Board* board;
extern Worm* worm;
extern Engine* engine;

// A cell on the game board
// -----------------------------------------------------------------------------

// default constructor
Cell::Cell()
     :Display_Object()
{
	this->free = false;

    this->cost = 10000;

    highlighted = false;
}

// constructor specifying the cell position
Cell::Cell(int x, int y)
     :Display_Object(x, y)
{
    this->cost = 0;

    highlighted = false;

    matShininess[0] = f2vt(20.0);
    setGLArray4(matSpecular, f2vt(0.7), f2vt(1.0), f2vt(0.8), f2vt(1.0));
    setGLArray(matAmbientAndDiff, f2vt(0.1), f2vt(0.3), f2vt(0.2));

    // __NEW__
        groundShininess[0] = f2vt(20.0);
        setGLArray4(groundSpecular, f2vt(1.0), f2vt(1.0), f2vt(1.0), f2vt(1.0));
        setGLArray(groundAmbientAndDiff, f2vt(0.3), f2vt(0.6), f2vt(0.6));

    free = true;
}

void Cell::highlightLight()
{
    highlighted = true;
    setGLArray4(groundSpecular, f2vt(1.0), f2vt(1.0), f2vt(1.0), f2vt(1.0));
    setGLArray(groundAmbientAndDiff, f2vt(0.6), f2vt(0.4), f2vt(0.6));
}

void Cell::highlight()
{
    highlighted = true;
    setGLArray4(groundSpecular, f2vt(1.0), f2vt(1.0), f2vt(1.0), f2vt(1.0));
    setGLArray(groundAmbientAndDiff, f2vt(0.4), f2vt(0.9), f2vt(0.9));
}

void Cell::resetHighlight()
{
    highlighted = false;;
    setGLArray4(groundSpecular, f2vt(1.0), f2vt(1.0), f2vt(1.0), f2vt(1.0));
    setGLArray(groundAmbientAndDiff, f2vt(0.3), f2vt(0.6), f2vt(0.6));
}

void Cell::highlightDark()
{
    highlighted = true;
    setGLArray4(groundSpecular, f2vt(1.0), f2vt(1.0), f2vt(1.0), f2vt(1.0));
    setGLArray(groundAmbientAndDiff, f2vt(0.3), f2vt(0.4), f2vt(0.4));
}

void Cell::highlightDarker()
{
    highlighted = true;
    setGLArray4(groundSpecular, f2vt(1.0), f2vt(1.0), f2vt(1.0), f2vt(1.0));
    setGLArray(groundAmbientAndDiff, f2vt(0.2), f2vt(0.2), f2vt(0.2));
}

// destroyer
Cell::~Cell(void)
{
}

// used to know whether the cell is free
bool Cell::isFree(void)
{
    return free;
}

// free the cell
void Cell::setFree(void)
{
    free = true;
}

// mark the cell as busy
void Cell::setUsed(void)
{
    free = false;
}

int Cell::getCost()
{
	return cost;
}

void Cell::computeCost()
{
	cost = __computeCost_internal();

	/*
	int ratio = cost / 10000;
	printf("%d\n", ratio);
	fflush(stdout);

    groundShininess[0] = f2vt(20.0 * ratio);
    setGLArray4(groundSpecular, f2vt(1.0 * ratio), f2vt(1.0 * ratio), f2vt(1.0 * ratio), f2vt(1.0));
    setGLArray(groundAmbientAndDiff, f2vt(0.3 * ratio), f2vt(0.6 * ratio), f2vt(0.6 * ratio));
    */

}

void Cell::printReachableCellsMarker (bool useThis) {

	bool** markers = engine->buildReachableCellsMarkerMarkers(useThis ? this : NULL);

	// finally if some cells in map have not been flagged : I have a partitioning
	printf("\n\n--> for cell (%d, %d) with using this ? %d\n\n", this->posX, this->posY, useThis);
	for (int i = 0; i < board->getSizeX(); i++) {
		for (int j = 0; j < board->getSizeY(); j++) {
			if (!markers[i][j]) {
				printf("  ");
			} else {
				printf("O ");
			}
		}
		printf("\n");
	}
	fflush (stdout);
	// cleanup
	for (int i = 0; i < board->getSizeX(); i++) {
		delete markers[i];
	}
	delete markers;
}

bool Cell::hasPartition(bool useThis) {

	bool hasPartition = false;

	// In this specific case, we have a risk of partitioning the board.
	// Let's find this out : easy in fact : Take the seed cell and ensure we can reach each and every other cell

	bool** markers = engine->buildReachableCellsMarkerMarkers(useThis ? this : NULL);
	// finally if some cells in map have not been flagged : I have a partitioning
	for (int i = 0; i < board->getSizeX(); i++) {
		for (int j = 0; j < board->getSizeY(); j++) {
			if (!markers[i][j]) {
				hasPartition = true;
				break;
			}
		}
		if (hasPartition) {
			break;
		}
	}
	// cleanup
	for (int i = 0; i < board->getSizeX(); i++) {
		delete markers[i];
	}
	delete markers;
	return hasPartition;
}


int Cell::__computeCost_internal()
{


	// FIXME get rid ot it
	this->resetHighlight();






    int cellX, cellY;
    this->getPosition(cellX, cellY);

    // put a huge price on cells where there actually is a body part
	if (!this->isFree())// worm->isPositionUnderBodyPart(cellX, cellY))
	{
		//this->highlightDarker();
		return 999999999; // TODO
	}

    int bodyPartNearCount = worm->getNearBodyPartCount(cellX, cellY);

    int boardXSize, boardYSize;
    boardXSize = board->getSizeX();
    boardYSize = board->getSizeY();


/*
 * Flawed !
 *
 * This doesn't work as expected !
 * Plus it causes performance issues
 */

/*
	// If putting the worm on this cell would cause a partitioning of the board, we should return a freaking big value
	if ( // only applies to cell on border
		 (cellY == 0 || cellY == boardYSize - 1 || cellX == 0 || cellX == boardXSize - 1)
		 // or cells with body part aside
		 ||
		 (bodyPartNearCount > 0)
		)


		// FIXME make it only for cells close to the worm head !!!
		// FIXME But debug it first !!!

	{
		bool hadPartitionWithout = hasPartition(false);
		bool hasPartitionWith = hasPartition(true);

		if (!hadPartitionWithout && hasPartitionWith)
		{

			// FIXME temp debugging
			printReachableCellsMarker (false);
			printReachableCellsMarker (true);


			this->highlightDark();
			return 200000;
		}
	}
*/


    // make it keep close to its own body
    if (bodyPartNearCount > 0)
    {
    	this->highlightLight();
        return 10000 / ((bodyPartNearCount * bodyPartNearCount) + 1); // dynamic cost
    }


    // and cell close to the border
    if (cellY == 0 || cellY == boardYSize - 1)
    {
    	this->highlight();
        return 9000;
    }
    if (cellX == 0 || cellX == boardXSize - 1)
    {
    	this->highlight();
        return 9000;
    }


    return 10000;
}

std::vector<Cell*>* Cell::getNeighbours()
{
    std::vector<Cell*>* retList = new std::vector<Cell*>();

    if (this->posX > 0)
    {
        retList->push_back(board->getCell(posX - 1, posY));
    }

    if (this->posY > 0)
    {
        retList->push_back(board->getCell(posX, posY - 1));
    }

    if (this->posX < BOARD_SIZE - 1)
    {
        retList->push_back(board->getCell(posX + 1, posY));
    }

    if (this->posY < BOARD_SIZE - 1)
    {
        retList->push_back(board->getCell(posX, posY + 1));
    }

    return retList;
}

direction_t Cell::getCellDirection (Cell* cell)
{
    if (posX < cell->posX)
    {
        return DIR_RIGHT;
    }
    else if (posX > cell->posX)
    {
        return DIR_LEFT;
    }
    else if (posY < cell->posY)
    {
        return DIR_UP;
    }
    else
    {
        return DIR_DOWN;
    }
}

// cell display func
void Cell::display(void)
{
    glPushMatrix();

        myGlTranslate(f2vt(100.0 * this->posX), f2vt(100.0 * this->posY), f2vt(0.0));

        // __NEW__
/*
        if (highlighted)
        {
            glPushMatrix();
                // draw the parcel square
                myGlMaterialv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, groundAmbientAndDiff);
                myGlMaterialv(GL_FRONT_AND_BACK, GL_SHININESS, groundShininess);
                myGlMaterialv(GL_FRONT_AND_BACK, GL_SPECULAR, groundSpecular);

                myGlTranslate(0, 0, f2vt(1.0));
                drawSquare();
            glPopMatrix();
        }
*/

        glPushMatrix();

			// draw the little square
			prepareDisplay(2.0);

			drawLittleSquare();

        glPopMatrix();

    glPopMatrix();
}

