// ============================================================================
//
// Copyright (c) 2010-2016, niceideas.ch - Jerome Kehrli
//
// You may distribute this code under the terms of the GNU LGPL license
// (http://www.gnu.org/licenses/lgpl.html). [^]
//
// ============================================================================



#include "geometry.h"


static MyGLReal boxvec[6][3] =
{
    {f2vt(-1.0), 0         , 0},
    {0         , f2vt( 1.0), 0},
    {f2vt( 1.0), 0         , 0},
    {0         , f2vt(-1.0), 0},
    {0         , 0         , f2vt( 1.0)},
    {0         , 0         , f2vt(-1.0)}
};

static GLushort boxndex [12][3] =
{
    {0, 1, 2},
    {0, 2, 3},
    {3, 2, 6},
    {3, 6, 7},
    {6, 4, 7},
    {6, 5, 4},
    {4, 5, 1},
    {4, 1, 0},
    {2, 1, 5},
    {2, 5, 6},
    {3, 7, 4},
    {3, 4, 0}
};

static GLushort wireboxndex[6][4] =
{
    {0, 1, 2, 3},
    {3, 2, 6, 7},
    {7, 6, 5, 4},
    {4, 5, 1, 0},
    {5, 6, 2, 1},
    {7, 4, 0, 3}
};

void setGLArray(MyGLReal* tab, MyGLReal red, MyGLReal green, MyGLReal blue)
{
    tab[0] = red;
    tab[1] = green;
    tab[2] = blue;
}

void setGLArray4(MyGLReal* tab, MyGLReal red, MyGLReal green,
        MyGLReal blue, MyGLReal alpha)
{
    tab[0] = red;
    tab[1] = green;
    tab[2] = blue;
    tab[3] = alpha;
}

void computeNormalVector(
        MyGLReal* point1, MyGLReal* point2, MyGLReal* point3, MyGLReal norm[3])
{
    MyGLReal vect1[] = {
            point1[0] - point2[0],
            point1[1] - point2[1],
            point1[2] - point2[2] };

    MyGLReal vect2[] = {
            point3[0] - point2[0],
            point3[1] - point2[1],
            point3[2] - point2[2] };

    norm[0] = f2vt(vt2f(vect1[1]) * vt2f(vect2[2]) - vt2f(vect1[2]) * vt2f(vect2[1]));
    norm[1] = f2vt(vt2f(vect1[2]) * vt2f(vect2[0]) - vt2f(vect1[0]) * vt2f(vect2[2]));
    norm[2] = f2vt(vt2f(vect1[0]) * vt2f(vect2[1]) - vt2f(vect1[1]) * vt2f(vect2[0]));
}


// find plane from thee points
void findPlane(float plane[4], float v0[3], float v1[3], float v2[3])
{
    float vec0[3], vec1[3];

    // need two vectors
    vec0[0] = v1[0] - v0[0];
    vec0[1] = v1[1] - v0[1];
    vec0[2] = v1[2] - v0[2];

    vec1[0] = v2[0] - v0[0];
    vec1[1] = v2[1] - v0[1];
    vec1[2] = v2[2] - v0[2];

    // vectorial product
    plane[0] = vec0[1] * vec1[2] - vec0[2] * vec1[1];
    plane[1] = vec0[2] * vec1[0] - vec0[0] * vec1[2];
    plane[2] = vec0[0] * vec1[1] - vec0[1] * vec1[0];

    plane[3] = -(plane[0] * v0[0] + plane[1] * v0[1] + plane[2] * v0[2]);
}


static void normalize(float v[3])
{
    MyGLReal r;

    r = sqrt( v[0]*v[0] + v[1]*v[1] + v[2]*v[2] );
    if (r == 0.0) return;

    v[0] /= r;
    v[1] /= r;
    v[2] /= r;
}

static void crossProduct(float v1[3], float v2[3], float result[3])
{
     result[0] = v1[1]*v2[2] - v1[2]*v2[1];
     result[1] = v1[2]*v2[0] - v1[0]*v2[2];
     result[2] = v1[0]*v2[1] - v1[1]*v2[0];
}

static void __gluMakeIdentityf(MyGLReal m[16])
{
     m[0+4*0] = f2vt(1); m[0+4*1] = f2vt(0); m[0+4*2] = f2vt(0); m[0+4*3] = f2vt(0);
     m[1+4*0] = f2vt(0); m[1+4*1] = f2vt(1); m[1+4*2] = f2vt(0); m[1+4*3] = f2vt(0);
     m[2+4*0] = f2vt(0); m[2+4*1] = f2vt(0); m[2+4*2] = f2vt(1); m[2+4*3] = f2vt(0);
     m[3+4*0] = f2vt(0); m[3+4*1] = f2vt(0); m[3+4*2] = f2vt(0); m[3+4*3] = f2vt(1);
}

void myGluLookAt(
        MyGLReal eyex, MyGLReal eyey, MyGLReal eyez,
        float centerx, float centery, float centerz,
        float upx, float upy, float upz)
{
    float forward[3], side[3], up[3];
    MyGLReal m[4][4];

    forward[0] = centerx - vt2f(eyex);
    forward[1] = centery - vt2f(eyey);
    forward[2] = centerz - vt2f(eyez);

    up[0] = upx;
    up[1] = upy;
    up[2] = upz;

    normalize(forward);

    /* Side = forward x up */
    crossProduct(forward, up, side);
    normalize(side);

    /* Recompute up as: up = side x forward */
    crossProduct(side, forward, up);

    __gluMakeIdentityf(&m[0][0]);
    m[0][0] = f2vt(side[0]);
    m[1][0] = f2vt(side[1]);
    m[2][0] = f2vt(side[2]);

    m[0][1] = f2vt(up[0]);
    m[1][1] = f2vt(up[1]);
    m[2][1] = f2vt(up[2]);

    m[0][2] = f2vt(-forward[0]);
    m[1][2] = f2vt(-forward[1]);
    m[2][2] = f2vt(-forward[2]);

    myGlMultMatrix(&m[0][0]);
    myGlTranslate(-eyex, -eyey, -eyez);
}

// find shadow projection matrix
void shadowMatrixCalc(MyGLReal shadowMat[4][4], float groundPlane[4],
        MyGLReal lightPos[4])
{
    // scalar product
    float dot = groundPlane[0] * vt2f(lightPos[0])
              + groundPlane[1] * vt2f(lightPos[1])
              + groundPlane[2] * vt2f(lightPos[2])
              + groundPlane[3] * vt2f(lightPos[3]);

    shadowMat[0][0] = f2vt(dot - vt2f(lightPos[0]) * groundPlane[0]);
    shadowMat[1][0] = f2vt(0.0 - vt2f(lightPos[0]) * groundPlane[1]);
    shadowMat[2][0] = f2vt(0.0 - vt2f(lightPos[0]) * groundPlane[2]);
    shadowMat[3][0] = f2vt(0.0 - vt2f(lightPos[0]) * groundPlane[3]);

    shadowMat[0][1] = f2vt(0.0 - vt2f(lightPos[1]) * groundPlane[0]);
    shadowMat[1][1] = f2vt(dot - vt2f(lightPos[1]) * groundPlane[1]);
    shadowMat[2][1] = f2vt(0.0 - vt2f(lightPos[1]) * groundPlane[2]);
    shadowMat[3][1] = f2vt(0.0 - vt2f(lightPos[1]) * groundPlane[3]);

    shadowMat[0][2] = f2vt(0.0 - vt2f(lightPos[2]) * groundPlane[0]);
    shadowMat[1][2] = f2vt(0.0 - vt2f(lightPos[2]) * groundPlane[1]);
    shadowMat[2][2] = f2vt(dot - vt2f(lightPos[2]) * groundPlane[2]);
    shadowMat[3][2] = f2vt(0.0 - vt2f(lightPos[2]) * groundPlane[3]);

    shadowMat[0][3] = f2vt(0.0 - vt2f(lightPos[3]) * groundPlane[0]);
    shadowMat[1][3] = f2vt(0.0 - vt2f(lightPos[3]) * groundPlane[1]);
    shadowMat[2][3] = f2vt(0.0 - vt2f(lightPos[3]) * groundPlane[2]);
    shadowMat[3][3] = f2vt(dot - vt2f(lightPos[3]) * groundPlane[3]);
}

void myDrawSquare(float size)
{
    MyGLReal point1[] = { 0               , 0               , 0 };
    MyGLReal point2[] = { f2vt((float)1.0 * size), 0               , 0 };
    MyGLReal point3[] = { 0               , f2vt((float)1.0 * size), 0 };
    MyGLReal point4[] = { f2vt((float)1.0 * size), f2vt((float)1.0 * size), 0 };

    // compute normal vector
    MyGLReal norm[3];
    computeNormalVector(point3, point2, point1, norm);

    const MyGLReal points[] =
    {
        point1[0], point1[1], point1[2],
        point2[0], point2[1], point2[2],
        point3[0], point3[1], point3[2],
        point4[0], point4[1], point4[2],
    };

    glEnableClientState (GL_VERTEX_ARRAY);
    glVertexPointer (3, MY_GL_REAL , 0, points);

    myGlNormal3(norm[0], norm[1], norm[2]);

    glDrawArrays (GL_TRIANGLE_STRIP, 0, 4);

    glDisableClientState (GL_VERTEX_ARRAY);
}

void drawSquare(void)
{
    myDrawSquare(100.0);
}

void drawLittleSquare(void)
{
    myGlTranslate(f2vt(45.0), f2vt(45.0), 0);
    myDrawSquare(10.0);
}

void drawBigSquare(void)
{
    myDrawSquare(100.0 * (float) BOARD_SIZE);
}

void mySolidTorus(float ir, float oor, GLint sides, GLint rings)
{
    GLint i, j, k, triangles;
    float s, t, x, y, z, twopi, nx, ny, nz;
    float sin_s, cos_s, cos_t, sin_t, twopi_s, twopi_t;
    float twopi_sides, twopi_rings;
    static MyGLReal *v, *n;
    static float parms[4];
    MyGLReal* p, *q;

    if (v)
    {
        if (parms[0] != ir || parms[1] != oor || parms[2] != sides || parms[3] != rings)
        {
            free(v);
            free(n);
            n = v = 0;

            glVertexPointer(3, MY_GL_REAL, 0, 0);
            glNormalPointer(MY_GL_REAL, 0, 0);
        }
    }

    if (!v)
    {
        parms[0] = ir;
        parms[1] = oor;
        parms[2] = (float)sides;
        parms[3] = (float)rings;

        p = v = (MyGLReal*)malloc(sides*(rings+1)*2*3*sizeof *v);
        q = n = (MyGLReal*)malloc(sides*(rings+1)*2*3*sizeof *n);

        twopi = 2.0f * (MyGLReal)MY_PI;
        twopi_sides = twopi/sides;
        twopi_rings = twopi/rings;

        for (i = 0; i < sides; i++)
        {
            for (j = 0; j <= rings; j++)
            {
                for (k = 1; k >= 0; k--)
                {
                    s = (i + k) % sides + 0.5f;
                    t = (float)( j % rings);

                    twopi_s= s*twopi_sides;
                    twopi_t = t*twopi_rings;

                    cos_s = (float)cos(twopi_s);
                    sin_s = (float)sin(twopi_s);

                    cos_t = (float)cos(twopi_t);
                    sin_t = (float)sin(twopi_t);

                    x = (oor+ir*(float)cos_s)*(float)cos_t;
                    y = (oor+ir*(float)cos_s)*(float)sin_t;
                    z = ir * (float)sin_s;

                    *p++ = f2vt(x);
                    *p++ = f2vt(y);
                    *p++ = f2vt(z);

                    nx = (float)cos_s*(float)cos_t;
                    ny = (float)cos_s*(float)sin_t;
                    nz = (float)sin_s;

                    *q++ = f2vt(nx);
                    *q++ = f2vt(ny);
                    *q++ = f2vt(nz);
                }
            }
        }
    }

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glNormalPointer(MY_GL_REAL, 0, n);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);

    triangles = (rings + 1) * 2;

    for(i = 0; i < sides; i++)
        glDrawArrays(GL_TRIANGLE_STRIP, triangles * i, triangles);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

}

void mySolidBox(MyGLReal Width, MyGLReal Depth, MyGLReal Height)
{
    int i;
    MyGLReal v[8][3];

    v[0][0] = v[1][0] = v[2][0] = v[3][0] = f2vt(- vt2f(Width) / 2.0f);
    v[4][0] = v[5][0] = v[6][0] = v[7][0] = f2vt(  vt2f(Width) / 2.0f);
    v[0][1] = v[1][1] = v[4][1] = v[5][1] = f2vt(- vt2f(Depth) / 2.0f);
    v[2][1] = v[3][1] = v[6][1] = v[7][1] = f2vt(  vt2f(Depth) / 2.0f);
    v[0][2] = v[3][2] = v[4][2] = v[7][2] = f2vt(- vt2f(Height) / 2.0f);
    v[1][2] = v[2][2] = v[5][2] = v[6][2] = f2vt(  vt2f(Height) / 2.0f);

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glEnableClientState (GL_VERTEX_ARRAY);

    for (i = 0; i < 6; i++)
    {
        myGlNormal3(boxvec[i][0], boxvec[i][1], boxvec[i][2]);
        glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, boxndex[i*2]);
    }

    glDisableClientState (GL_VERTEX_ARRAY);
}


void myWireBox(MyGLReal Width, MyGLReal Depth, MyGLReal Height)
{
    MyGLReal v[8][3];
    int i;

    v[0][0] = v[1][0] = v[2][0] = v[3][0] = f2vt(- vt2f(Width) / 2.0f);
    v[4][0] = v[5][0] = v[6][0] = v[7][0] = f2vt(  vt2f(Width) / 2.0f);
    v[0][1] = v[1][1] = v[4][1] = v[5][1] = f2vt(- vt2f(Depth) / 2.0f);
    v[2][1] = v[3][1] = v[6][1] = v[7][1] = f2vt(  vt2f(Depth) / 2.0f);
    v[0][2] = v[3][2] = v[4][2] = v[7][2] = f2vt(- vt2f(Height) / 2.0f);
    v[1][2] = v[2][2] = v[5][2] = v[6][2] = f2vt(  vt2f(Height) / 2.0f);

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glEnableClientState (GL_VERTEX_ARRAY);

    for ( i = 0; i < 6; i++)
    {
        myGlNormal3(boxvec[i][0], boxvec[i][1], boxvec[i][2]);
        glDrawElements(GL_LINE_LOOP, 4, GL_UNSIGNED_SHORT, wireboxndex[i]);
    }
    glDisableClientState (GL_VERTEX_ARRAY);
}

void _plotSpherePoints(float radius, GLint stacks, GLint slices, MyGLReal* v, MyGLReal* n)
{

    GLint i, j;
    float slicestep, stackstep;

    stackstep = ((float)MY_PI) / stacks;
    slicestep = 2.0f * ((float)MY_PI) / slices;

    for (i = 0; i < stacks; ++i)
    {
        float a = i * stackstep;
        float b = a + stackstep;

        float s0 =  (float)sin(a);
        float s1 =  (float)sin(b);

        float c0 =  (float)cos(a);
        float c1 =  (float)cos(b);

        for (j = 0; j <= slices; ++j)
        {
            float c = j * slicestep;
            float x = (float)cos(c);
            float y = (float)sin(c);

            *n = f2vt(x * s0);
            *v = f2vt(vt2f(*n) * radius);

            n++;
            v++;

            *n = f2vt(y * s0);
            *v = f2vt(vt2f(*n) * radius);

            n++;
            v++;

            *n = f2vt(c0);
            *v = f2vt(vt2f(*n) * radius);

            n++;
            v++;

            *n = f2vt(x * s1);
            *v = f2vt(vt2f(*n) * radius);

            n++;
            v++;

            *n = f2vt(y * s1);
            *v = f2vt(vt2f(*n) * radius);

            n++;
            v++;

            *n = f2vt(c1);
            *v = f2vt(vt2f(*n) * radius);

            n++;
            v++;

        }
    }
}

void mySolidSphere(float radius, GLint slices, GLint stacks)
{
    GLint i, triangles;
    static MyGLReal* v, *n;
    static float parms[3];

    if (v)
    {
        if (parms[0] != radius || parms[1] != slices || parms[2] != stacks)
        {
            free(v);
            free(n);

            n = v = 0;

            glVertexPointer(3, MY_GL_REAL, 0, 0);
            glNormalPointer(MY_GL_REAL, 0, 0);
        }
    }

    if (!v)
    {
        parms[0] = radius;
        parms[1] = (float)slices;
        parms[2] = (float)stacks;

        v = (MyGLReal*)malloc(stacks*(slices+1)*2*3*sizeof *v);
        n = (MyGLReal*)malloc(stacks*(slices+1)*2*3*sizeof *n);

        _plotSpherePoints(radius, stacks, slices, v, n);

    }

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glNormalPointer(MY_GL_REAL, 0, n);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);

    triangles = (slices + 1) * 2;

    for(i = 0; i < stacks; i++)
        glDrawArrays(GL_TRIANGLE_STRIP, i * triangles, triangles);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

}

void myWireSphere(float radius, GLint slices, GLint stacks)
{
    GLint i, j, f;
    static MyGLReal* v, *n;
    static float parms[3];

    if (v)
    {
        if (parms[0] != radius || parms[1] != slices || parms[2] != stacks)
        {
            free(v);
            free(n);

            n = v = 0;

            glVertexPointer(3, MY_GL_REAL, 0, 0);
            glNormalPointer(MY_GL_REAL, 0, 0);
        }
    }

    if (!v)
    {
        parms[0] = radius;
        parms[1] = (float)slices;
        parms[2] = (float)stacks;

        v = (MyGLReal*)malloc(stacks*(slices+1)*2*3*sizeof *v);
        n = (MyGLReal*)malloc(stacks*(slices+1)*2*3*sizeof *n);

        _plotSpherePoints(radius, stacks, slices, v, n);

    }

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glNormalPointer(MY_GL_REAL, 0, n);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);

    for(i = 0; i < stacks; ++i)
    {
        f = i * (slices + 1);

        for (j = 0; j <= slices; ++j)
            glDrawArrays(GL_LINE_LOOP, (f + j)*2, 3);
    }

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

}

void mySolidCone(float base, float height, GLint slices, GLint stacks)
{
    GLint i, j;
    float twopi, nx, ny, nz;
    static MyGLReal* v, *n;
    static float parms[4];
    MyGLReal* p, *q;

    if (v)
    {
        if (parms[0] != base || parms[1] != height || parms[2] != slices || parms[3] != stacks)
        {
            free(v);
            free(n);

            n = v = 0;

            glVertexPointer(3, MY_GL_REAL, 0, 0);
            glNormalPointer(MY_GL_REAL, 0, 0);
        }
    }

    if ((!v) && (height != 0.0f))
    {
        float phi = (float)atan(base/height);
        float cphi = (float)cos(phi);
        float sphi= (float)sin(phi);

        parms[0] = base;
        parms[1] = height;
        parms[2] = (float)slices;
        parms[3] = (float)stacks;

        p = v = (MyGLReal*)malloc(stacks*(slices+1)*2*3*sizeof *v);
        q = n = (MyGLReal*)malloc(stacks*(slices+1)*2*3*sizeof *n);

        twopi = 2.0f * ((float)MY_PI);

        for (i = 0; i < stacks; i++)
        {
            float r = base*(1.0f - (float)i /stacks);
            float r1 = base*(1.0f - (float)(i+1.0)/stacks);
            float z = height*i /stacks;
            float z1 = height*(1.0f+i) /stacks;

            for (j = 0; j <= slices; j++)
            {
                float theta = j == slices ? 0.f : (float) j /slices*twopi;
                float ctheta = (float)cos(theta);
                float stheta = (float)sin(theta);

                nx = ctheta;
                ny = stheta;
                nz = sphi;

                *p++ = f2vt(r1*nx);
                *p++ = f2vt(r1*ny);
                *p++ = f2vt(z1);

                *q++ = f2vt(nx*cphi);
                *q++ = f2vt(ny*cphi);
                *q++ = f2vt(nz);

                *p++ = f2vt(r*nx);
                *p++ = f2vt(r*ny);
                *p++ = f2vt(z);


                *q++ = f2vt(nx*cphi);
                *q++ = f2vt(ny*cphi);
                *q++ = f2vt(nz);

            }
        }
    }

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glNormalPointer(MY_GL_REAL, 0, n);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);

    for(i = 0; i < stacks; i++)
        glDrawArrays(GL_TRIANGLE_STRIP, i*(slices+1)*2, (slices+1)*2);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);
}

void mySolidCube(float size)
{
    static MyGLReal v[108];       // 108 =  6*18

    static const float cubev[108] =
    {
        -1., -1., 1.,    /* front */
         1., -1., 1.,
        -1.,  1., 1.,

         1., -1., 1.,
         1.,  1., 1.,
        -1.,  1., 1.,

        -1.,  1., -1.,    /* back */
         1., -1., -1.,
        -1., -1., -1.,

        -1.,  1., -1.,
         1.,  1., -1.,
         1., -1., -1.,

        -1., -1., -1.,    /* left */
        -1., -1.,  1.,
        -1.,  1., -1.,

        -1., -1.,  1.,
        -1.,  1.,  1.,
        -1.,  1., -1.,

         1., -1.,  1.,    /* right */
         1., -1., -1.,
         1.,  1.,  1.,

         1., -1., -1.,
         1.,  1., -1.,
         1.,  1.,  1.,

        -1.,  1.,  1.,    /* top */
         1.,  1.,  1.,
        -1.,  1., -1.,

         1.,  1.,  1.,
         1.,  1., -1.,
        -1.,  1., -1.,

        -1., -1., -1.,    /* bottom */
         1., -1., -1.,
        -1., -1.,  1.,

         1., -1., -1.,
         1., -1.,  1.,
        -1., -1.,  1.,
    };

    static const MyGLReal cuben[108] =
    {
        f2vt(0.), f2vt(0.), f2vt(1.),    /* front */
        f2vt(0.), f2vt(0.), f2vt(1.),
        f2vt(0.), f2vt(0.), f2vt(1.),

        f2vt(0.), f2vt(0.), f2vt(1.),
        f2vt(0.), f2vt(0.), f2vt(1.),
        f2vt(0.), f2vt(0.), f2vt(1.),

        f2vt(0.), f2vt(0.), -f2vt(1.),    /* back */
        f2vt(0.), f2vt(0.), -f2vt(1.),
        f2vt(0.), f2vt(0.), -f2vt(1.),

        f2vt(0.), f2vt(0.), -f2vt(1.),
        f2vt(0.), f2vt(0.), -f2vt(1.),
        f2vt(0.), f2vt(0.), -f2vt(1.),

        -f2vt(1.), f2vt(0.), f2vt(0.),    /* left */
        -f2vt(1.), f2vt(0.), f2vt(0.),
        -f2vt(1.), f2vt(0.), f2vt(0.),

        -f2vt(1.), f2vt(0.), f2vt(0.),
        -f2vt(1.), f2vt(0.), f2vt(0.),
        -f2vt(1.), f2vt(0.), f2vt(0.),

        f2vt(1.), f2vt(0.), f2vt(0.),    /* right */
        f2vt(1.), f2vt(0.), f2vt(0.),
        f2vt(1.), f2vt(0.), f2vt(0.),

        f2vt(1.), f2vt(0.), f2vt(0.),
        f2vt(1.), f2vt(0.), f2vt(0.),
        f2vt(1.), f2vt(0.), f2vt(0.),

        f2vt(0.), f2vt(1.), f2vt(0.),    /* top */
        f2vt(0.), f2vt(1.), f2vt(0.),
        f2vt(0.), f2vt(1.), f2vt(0.),

        f2vt(0.), f2vt(1.), f2vt(0.),
        f2vt(0.), f2vt(1.), f2vt(0.),
        f2vt(0.), f2vt(1.), f2vt(0.),

        f2vt(0.), -f2vt(1.), f2vt(0.),    /* bottom */
        f2vt(0.), -f2vt(1.), f2vt(0.),
        f2vt(0.), -f2vt(1.), f2vt(0.),

        f2vt(0.), -f2vt(1.), f2vt(0.),
        f2vt(0.), -f2vt(1.), f2vt(0.),
        f2vt(0.), -f2vt(1.), f2vt(0.),
    };

    int i;
    size /= 2;

    for(i = 0; i < 108; i++)
        v[i] = cubev[i] * size;

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glNormalPointer(MY_GL_REAL, 0, cuben);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);

    glDrawArrays(GL_TRIANGLES, 0, 36);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

}

void myWireCube(float size)
{
    static MyGLReal v[72];

    static const float cubev[72] =       // 72 = 3*6*4
    {
        -1., -1., 1.,    /* front */
        1., -1., 1.,
        1.,  1., 1.,
        -1.,  1., 1.,

        -1.,  1., -1.,    /* back */
        1.,  1., -1.,
        1., -1., -1.,
        -1., -1., -1.,

        -1., -1., -1.,    /* left */
        -1., -1.,  1.,
        -1.,  1.,  1.,
        -1.,  1., -1.,

        1., -1.,  1.,    /* right */
        1., -1., -1.,
        1.,  1., -1.,
        1.,  1.,  1.,

        -1.,  1.,  1.,    /* top */
        1.,  1.,  1.,
        1.,  1., -1.,
        -1.,  1., -1.,

        -1., -1., -1.,    /* bottom */
        1., -1., -1.,
        1., -1.,  1.,
        -1., -1.,  1.,
    };

    static const MyGLReal cuben[72] =
    {
        f2vt(0.), f2vt(0.), f2vt(1.),    /* front */
        f2vt(0.), f2vt(0.), f2vt(1.),
        f2vt(0.), f2vt(0.), f2vt(1.),
        f2vt(0.), f2vt(0.), f2vt(1.),

        f2vt(0.), f2vt(0.), -f2vt(1.),    /* back */
        f2vt(0.), f2vt(0.), -f2vt(1.),
        f2vt(0.), f2vt(0.), -f2vt(1.),
        f2vt(0.), f2vt(0.), -f2vt(1.),

        -f2vt(1.), f2vt(0.), f2vt(0.),    /* left */
        -f2vt(1.), f2vt(0.), f2vt(0.),
        -f2vt(1.), f2vt(0.), f2vt(0.),
        -f2vt(1.), f2vt(0.), f2vt(0.),

        f2vt(1.), f2vt(0.), f2vt(0.),    /* right */
        f2vt(1.), f2vt(0.), f2vt(0.),
        f2vt(1.), f2vt(0.), f2vt(0.),
        f2vt(1.), f2vt(0.), f2vt(0.),

        f2vt(0.), f2vt(1.), f2vt(0.),    /* top */
        f2vt(0.), f2vt(1.), f2vt(0.),
        f2vt(0.), f2vt(1.), f2vt(0.),
        f2vt(0.), f2vt(1.), f2vt(0.),

        f2vt(0.), -f2vt(1.), f2vt(0.),    /* bottom */
        f2vt(0.), -f2vt(1.), f2vt(0.),
        f2vt(0.), -f2vt(1.), f2vt(0.),
        f2vt(0.), -f2vt(1.), f2vt(0.),
    };

    int i;
    size /= 2;

    for(i = 0; i < 72; i++)
        v[i] = cubev[i] * size;

    glVertexPointer(3, MY_GL_REAL, 0, v);
    glNormalPointer(MY_GL_REAL, 0, cuben);

    glEnableClientState (GL_VERTEX_ARRAY);
    glEnableClientState (GL_NORMAL_ARRAY);

    for(i = 0; i < 6; i++)
        glDrawArrays(GL_LINE_LOOP, 4*i, 4);

    glDisableClientState(GL_VERTEX_ARRAY);
    glDisableClientState(GL_NORMAL_ARRAY);

}


