// ============================================================================
//
// Copyright (c) 2010-2016, niceideas.ch - Jerome Kehrli
//
// You may distribute this code under the terms of the GNU LGPL license
// (http://www.gnu.org/licenses/lgpl.html). [^]
//
// ============================================================================



#include "engine.h"


// import global variables
extern Board* board;
extern Worm* worm;


void dijkstraComputePaths(vertex_t source,
                          adjacency_map_t& adjacencyMap,
                          std::map<vertex_t, weight_t>& minDistance,
                          std::map<vertex_t, vertex_t>& previous)
{
    for (adjacency_map_t::iterator vertexIter = adjacencyMap.begin();
         vertexIter != adjacencyMap.end();
         vertexIter++)
    {
        vertex_t v = vertexIter->first;
        //min_distance[v] = std::numeric_limits< int >::infinity();
        minDistance[v] = DOUBLE_MAX_VALUE;
    }

    minDistance[source] = 0;
    std::set< std::pair<weight_t, vertex_t>,
              pair_first_less<weight_t, vertex_t> > vertexQueue;
    for (adjacency_map_t::iterator vertexIter = adjacencyMap.begin();
         vertexIter != adjacencyMap.end();
         vertexIter++)
    {
        vertex_t v = vertexIter->first;
        vertexQueue.insert(std::pair<weight_t, vertex_t>(minDistance[v], v));
    }

    while (!vertexQueue.empty())
    {
        std::set< std::pair<weight_t, vertex_t> >::iterator itBegin = vertexQueue.begin();

        vertex_t u = itBegin->second;
        vertexQueue.erase(itBegin);


        // Visit each edge exiting u
        for (std::list<edge>::iterator edgeIter = adjacencyMap[u].begin();
             edgeIter != adjacencyMap[u].end();
             edgeIter++)
        {
            vertex_t v = edgeIter->target;
            weight_t weight = edgeIter->weight;
            weight_t distanceThroughU = minDistance[u] + weight;
            if (distanceThroughU < minDistance[v]) {
                vertexQueue.erase(std::pair<weight_t, vertex_t>(minDistance[v], v));

                minDistance[v] = distanceThroughU;
                previous[v] = u;

                vertexQueue.insert(std::pair<weight_t, vertex_t>(minDistance[v], v));
            }
        }
    }
}

std::list<vertex_t> dijkstraGetShortestPathTo(
    vertex_t target, std::map<vertex_t, vertex_t>& previous)
{
    std::list<vertex_t> path;
    std::map<vertex_t, vertex_t>::iterator prev;
    vertex_t vertex = target;
    path.push_front(vertex);

    while((prev = previous.find(vertex)) != previous.end())
    {
        vertex = prev->second;
        path.push_front(vertex);
    }
    return path;
}


// navigation engine class
// ----------------------------------------------------------------------------

// default constructor
Engine::Engine(void)
{
	askedDirection = DIR_UP;
    activ = true;
}

// reinit the engine
void Engine::init(void)
{
}

// activate / decativate auto-pilot
void Engine::switchEngineOnOff(void)
{
    activ = !activ;
}

void Engine::setAutopilot(void)
{
    activ = true;
}

void Engine::setManual(void)
{
    activ = false;
}

bool** Engine::buildReachableCellsMarkerMarkers(Cell* optionalCandidate)
{
	// Loop on all other cell and put them in a map associating a boolean
	// (OPTIMIZATION use an array of bool[][] instead
	bool** markers = new bool*[board->getSizeX()];
	for (int i = 0; i < board->getSizeX(); i++)
	{
		markers[i] = new bool[board->getSizeY()];
		for (int j = 0; j < board->getSizeY(); j++)
		{
			markers[i][j] = false;
		}
	}


	// Then do DFS taking into account busy cells as well as this very cell (as if it was busy)
	// - this DFS algo can be iterative and is fast = and flag encountered cells
	std::list<Cell*>* retList = new std::list<Cell*>();

	//  - initializing with seed cell
	Cell* seedCell = board->getSeedCell();
	//  - mark current as visited
	int curX, curY;
	seedCell->getPosition(curX, curY);
	markers[curX][curY] = true;
	retList->push_back (seedCell);

	//  - now we start
	while (!retList->empty())
	{

		Cell* current = retList->back();
		retList->pop_back();



		//   - get all neighbours
		std::vector<Cell*>* neighbours = current->getNeighbours();
		for (std::vector<Cell*>::iterator neighIt = neighbours->begin(); neighIt != neighbours->end(); neighIt++)
		{
			Cell* neighbour = *neighIt;

			int neiX, neiY;
			neighbour->getPosition(neiX, neiY);

			//   - If neighbour is already visited, then skip it
			//   - Otherwise analyze it further
			if (!markers[neiX][neiY])
			{

				if (neighbour->isFree() && (optionalCandidate == NULL || optionalCandidate != neighbour)) {
					retList->push_front (neighbour);
				}

				markers[neiX][neiY] = true;
			}
		}

		//  - cleanup
		delete neighbours;

	}

	// I need another loop since in the end if an unmarked cell has a body part on it, then it's not a partition
	for (int i = 0; i < board->getSizeX(); i++) {
		for (int j = 0; j < board->getSizeY(); j++) {
			if (!markers[i][j]) {
				Cell* cell = board->getCell(i, j);
				if (!cell->isFree() || (optionalCandidate != NULL && optionalCandidate == cell)) {
					markers[i][j] = true;
				}
			}
		}
	}

	delete retList;

	return markers;
}

// Get the next direction to be done
direction_t Engine::getNext(void)
{
    if (!activ)
    {
        return this->askedDirection;
    }

    this->init();

    int headX;
    int headY;
    worm->getHeadPosition(headX, headY);
    Cell* headCell = board->getCell(headX, headY);

    std::list<Cell*>* cells = board->getCellList();

    // 0. Compute all Costs once and for all
    for (std::list<Cell*>::iterator cellIt = cells->begin(); cellIt != cells->end(); cellIt++)
    {
        Cell* current = *cellIt;

        // Compute costs once and for all there !
        current->computeCost();

    }



    /*
     * FIXME TEMPORARY DEBUGGING SHIT !!!
     */
/*
    bool** markers = buildAndGetPartitionMarkers();

	// finally if some cells in map have not been flagged : I have a partitioning
	for (int i = 0; i < board->getSizeX(); i++)
	{
		for (int j = 0; j < board->getSizeY(); j++)
		{
			if (!markers[i][j])
			{
				board->getCell(i, j)->highlight();
			}
			else
			{
				board->getCell(i, j)->resetHighlight();
			}
		}

		// cleanup
		delete markers[i];
	}

	// cleanup
	delete markers;
*/

    /*
     * FIXME TEMPORARY DEBUGGING SHIT !!!
     */





    // 1. Build graph structure

    adjacency_map_t adjacencyMap;

    // for each cells
    for (std::list<Cell*>::iterator cellIt = cells->begin(); cellIt != cells->end(); cellIt++)
    {
        Cell* current = *cellIt;

        // get each neighbour
        std::vector<Cell*>* neighbours = current->getNeighbours();
        for (std::vector<Cell*>::iterator neighIt = neighbours->begin(); neighIt != neighbours->end(); neighIt++)
        {
            Cell* neighbour = *neighIt;

            // FIXME Optimization : compute neighbour->getCose only if not done (use an array as memory of cost(

            adjacencyMap[current].push_back(edge(neighbour, 100.0 * (double) neighbour->getCost()));
        }
    }

    // Got to clean it before delete it. Otherwise cells are deleted as well
    cells->clear();
    delete cells;

    std::map<vertex_t, weight_t> minDistance;
    std::map<vertex_t, vertex_t> previous;
    dijkstraComputePaths(headCell, adjacencyMap, minDistance, previous);

    Cell* seedCell = board->getSeedCell();

    double distance = minDistance[seedCell];
    bool noPathAvailable = distance == DOUBLE_MAX_VALUE;
    //std::cout << "Distance " << distance << " - path available ? " << noPathAvailable << std::endl;

    std::list<Cell*> cellTravelList;

    if (!noPathAvailable)
    {
        cellTravelList = dijkstraGetShortestPathTo(seedCell, previous);
    }
    else
    {
        // find longest path != than DOUBLE_MAX_VALUE
        double longestDistance = -1;
        vertex_t targetCell = NULL;
        for (std::map<vertex_t, weight_t>::iterator vertexDistIt = minDistance.begin(); vertexDistIt != minDistance.end(); vertexDistIt++)
        {
            vertex_t cell = vertexDistIt->first;
            double cellDist = vertexDistIt->second;
            if (cellDist != DOUBLE_MAX_VALUE)
            {
                noPathAvailable = false;
                if (cellDist > longestDistance)
                {
                    longestDistance = cellDist;
                    targetCell = cell;
                }
            }
        }
        if (targetCell != NULL)
        {
            cellTravelList = dijkstraGetShortestPathTo(targetCell, previous);
        }

    }

    if (!noPathAvailable)
    {
        //if (cell_travel_list != NULL && !cell_travel_list->empty())
        if (cellTravelList.size() >= 2) // we need the second
        {

            // 2. If a path could be computed and debugging is on
            //    then highlight the path

            /*
#ifdef DEBUG
            board->resetAllCellsHighlights ();
            for (std::list<Cell*>::iterator dbgIterator = cellTravelList.begin(); dbgIterator != cellTravelList.end(); dbgIterator++)
            {
                Cell* nextCell = *dbgIterator;
                nextCell->highlight();
            }
#endif
*/

            // 3. Return next direction according to path
            std::list<Cell*>::iterator iterator = cellTravelList.begin();
            iterator++; // we want the second cell in path
            direction_t retDirection = headCell->getCellDirection (*iterator);
            //delete cell_travel_list;
            setAskedDirection (retDirection);
            return retDirection;
        }
    }

    // 3. If that list is empty, get the list of directions to be followed to
    //    make the longest possible travel
    // TODO

    // (In the meantime use get dumb)
    direction_t retDirection = this->getDumbDirection();
    setAskedDirection (retDirection);
    return retDirection;

}

// dummy way to get a direction in case the worm is trapped
direction_t Engine::getDumbDirection(void)
{
    static direction_t dir = DIR_DOWN;
    static int dumbCount = 0;
    int headX, headY;
    bool doLoop;

    do
    {
        if (dumbCount > 5)
        {
            dumbCount = 0;
            return DIR_DOWN; // fuck !!
        }

        doLoop = false;

        if (dir == DIR_DOWN)
            dir = DIR_RIGHT;
        else if (dir == DIR_RIGHT)
            dir = DIR_UP;
        else if (dir == DIR_UP)
            dir = DIR_LEFT;
        else if (dir == DIR_LEFT)
            dir = DIR_DOWN;

        worm->getHeadPosition(headX, headY);
        switch (dir)
        {
        case DIR_RIGHT:
            if ((headX + 1 > board->getSizeX() - 1) || (!board->getCell(headX + 1, headY)->isFree()))
                doLoop = true;
            break;
        case DIR_LEFT:
            if ((headX - 1 < 0)                     || (!board->getCell(headX - 1, headY)->isFree()))
                doLoop = true;
            break;
        case DIR_UP:
            if ((headY + 1 > board->getSizeY() - 1) || (!board->getCell(headX, headY + 1)->isFree()))
                doLoop = true;
            break;
        case DIR_DOWN:
            if ((headY - 1 < 0)                     || (!board->getCell(headX, headY - 1)->isFree()))
                doLoop = true;
            break;
        }

        dumbCount++;
    }
    while (doLoop);

    dumbCount = 0;
    return dir;
}

void Engine::setAskedDirection (direction_t askedDirection)
{
    if (this->askedDirection == DIR_UP && askedDirection == DIR_DOWN)
    {
        return;
    }
    if (this->askedDirection == DIR_DOWN && askedDirection == DIR_UP)
    {
        return;
    }
    if (this->askedDirection == DIR_LEFT && askedDirection == DIR_RIGHT)
    {
        return;
    }
    if (this->askedDirection == DIR_RIGHT && askedDirection == DIR_LEFT)
    {
        return;
    }
    this->askedDirection = askedDirection;
}

direction_t Engine::getAskedDirection(void)
{
    return askedDirection;
}

bool Engine::isManual()
{
    return !activ;
}


