// ============================================================================
//
// Copyright (c) 2010-2016, niceideas.ch - Jerome Kehrli
//
// You may distribute this code under the terms of the GNU LGPL license
// (http://www.gnu.org/licenses/lgpl.html). [^]
//
// ============================================================================



#ifdef USE_QT
#include <QApplication>
#include <QMainWindow>
#include "mainwindow.h"
#endif

#include "common.h"

#include "engine.h"
#include "camera.h"
#include "worm.h"
#include "cell.h"
#include "board.h"
#include "spot.h"
#include "head.h"
#include "body_part.h"


Board* board = NULL; // the board
Worm* worm = NULL; // the worm
Camera* cam = NULL; // the camera
Spot* spot = NULL; // the spot
Engine* engine = NULL; // the navigation engine

// motion management
unsigned long startTime = 0;
unsigned long displayTime = 0;


int screenResX = SCREEN_RES_X;
int screenResY = SCREEN_RES_Y;

// three points in the plan
float v0[] = { -100.0, -100.0, 0.0 };
float v1[] = { 100.0, -100.0, 0.0 };
float v2[] = { 100.0, 100.0, 0.0 };

// shadow matrices
MyGLReal light0Shadow[4][4];

// the board plane
float plane[4];

MyGLReal ambientLight[] = { f2vt(0.4), f2vt(0.4), f2vt(0.4), f2vt(1.0) };
MyGLReal fogColor[] = { 0, 0, 0 };

MyGLReal lightPosition[3];


// keyboard management callback (common for glut and QT)
void keyboard(unsigned char key, int x, int y)
{

    //printf ("%i", key);
    switch (key) {
    case 27:
        exit(0);
        break;
    case 'P':
    case 'p':
    	worm->switchPause();
    	/* no break */
    case 'S':
    case 's':
        cam->switchRotationOnOff();
        break;
    case 'O':
    case 'o':
        engine->switchEngineOnOff();
        break;
    case 'h':
        engine->setAskedDirection (DIR_LEFT);
        break;
    case 'k':
        engine->setAskedDirection (DIR_RIGHT);
        break;
    case 'u':
        engine->setAskedDirection (DIR_UP);
        break;
    case 'j':
        engine->setAskedDirection (DIR_DOWN);
        break;
    }
}


void initializeGLImpl()
{

    spot->getLightPosition(lightPosition);

    // light 0 (single spot)
    spot->initSpot();

    myGlClearColor(0, 0, 0, 0);
    glShadeModel(GL_SMOOTH);

    myGlLightModelv(GL_LIGHT_MODEL_AMBIENT, ambientLight);

    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHTING);

    glShadeModel(GL_SMOOTH);


    glEnable(GL_DEPTH_TEST);
    //glEnable(GL_CULL_FACE);
    glEnable(GL_NORMALIZE);
    glDepthFunc(GL_LEQUAL);


    // initialize shadow
    glEnable(GL_FOG);
    {
        myGlFog(GL_FOG_MODE, GL_LINEAR);
        myGlFogv(GL_FOG_COLOR, fogColor);
        myGlFog(GL_FOG_DENSITY, f2vt(0.15));
        glHint(GL_FOG_HINT, GL_DONT_CARE);
        myGlFog(GL_FOG_START, f2vt(30.0 * BOARD_SIZE));
        myGlFog(GL_FOG_END, f2vt(140.0 * BOARD_SIZE));
    }

    // build matrix used to apply shadow
    findPlane(plane, v0, v1, v2);
    shadowMatrixCalc(light0Shadow, plane, lightPosition);

}


unsigned long getMotionRealtimeSleepTime()
{
    // display_time is the time required for the last display

     // 1. First, let's make sure we'll have exactly 20 display every second

     // If each display takes display_time ms, then there is
     // tot_disp_time_p_sec = (1000 ms - 20 * display_time ms)
     // available for sleeping. So we sleep (tot_disp_time_p_sec / 20) ms
     unsigned long totDisplayTimeBySec = 1000 - (DISPLAY_EACH_SECOND * displayTime);
     unsigned long sleepTime = totDisplayTimeBySec / DISPLAY_EACH_SECOND;

     // 2. Then let's compute the worm move coefficient

     // Initially, we want the worm to move 3 case each seconds
     // We have 20 display per seconds
     // 3 cases means 300 offset
     // This means it needs to do 300 / 20 = 15 offset each display

     // BUT : we want him to go twice as fast when it reaches 20 body parts,
     // hence :

     float speed = WORM_INITIAL_SPEED_OFFSET
             + (WORM_INITIAL_SPEED_OFFSET * worm->getBodyPartCount() / 20);

     // but max is 2.2 times as fast as initial time
     float maxSpeed = 2.2 * WORM_INITIAL_SPEED_OFFSET;
     if (speed > maxSpeed)
     {
         speed = maxSpeed;
     }
     worm->fixeCoeff(speed);


     //printf("Sleeping %lu ms  \n", sleep_time);
     if(sleepTime < 1000 / DISPLAY_EACH_SECOND)
     {
         //usleep(sleep_time * 1000);
         return sleepTime;
     }
     return 0;
}

void displayGame()
{

    cam->reshape(screenResX, screenResY);

    glPushMatrix();

        // clear current buffer
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        //glLoadIdentity(); // reset transformations

        // position camera and spot
        spot->positionate();
        cam->positionate();

        glPushMatrix();

            myGlTranslate(f2vt(-50.0 * board->getSizeX()), f2vt(-50.0 * board->getSizeY()), 0);

            // display board and worm
            board->display();
            worm->display();

        glPopMatrix();

    glPopMatrix();
}

void initGame(void)
{

    board->init();
    worm->init();
    engine->init();
    cam->init();

    board->createRandomSeed();

    // TODO
    worm->setDirection(DIR_UP);
    //worm->set_direction(engine->get_next());
}

void gameMotion()
{
    startTime = getTime();

    worm->move();

    // change cell in case the offset has been exceeded
    if (worm->getOffset() >= OFFSET_RANGE)
    {
        if (!worm->moveCell())
        {
            // re-init game in case the worm crashed doing the movement
            initGame();
        }
    }
}

#ifdef USE_GLUT

void motion(void)
{
    unsigned long sleepTime = getMotionRealtimeSleepTime();

    //printf("Sleeping %lu ms  \n", sleepTime);

    if(sleepTime != 0)
    {
        usleep(sleepTime * 1000);
    }

    gameMotion();

    // call for redisplay
    glutPostRedisplay();
}

// display management callback
void display(void)
{
    displayGame();

    // NOW really do display !
    glutSwapBuffers();

    unsigned long endTime = getTime();
    displayTime = endTime - startTime;

}


// Used by the glut
void reshape(int width, int height)
{
    screenResX = width;
    screenResY = height;
    cam->reshape(width, height);
}

#endif


int main( int argc, char ** argv )
{

    // initialize random number generator // TODO find something better
    srand((unsigned) getTime() + rand());

    board = new Board(BOARD_SIZE, BOARD_SIZE);
    worm = new Worm(BOARD_SIZE / 2, BOARD_SIZE / 2, DIR_UP);
    cam = new Camera;

    engine = new Engine();
    initGame();
    spot = new Spot;

#ifdef USE_QT
    QApplication a( argc, argv );
    MainWindow mw;
    //mw.showMaximized();
    mw.resize (800, 600);
    mw.show();

    //a.installEventFilter(&mw);

    return a.exec();
#endif


#ifdef USE_GLUT
    // glut is used for window management
    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA);
    glutInitWindowPosition(0, 0);
    glutInitWindowSize(SCREEN_RES_X, SCREEN_RES_Y);
    glutCreateWindow(WINDOW_TITLE);

    // initialisation
    initializeGLImpl();

    // register callbacks
    glutKeyboardFunc(keyboard);
    glutDisplayFunc(display);
    glutReshapeFunc(reshape);
    glutIdleFunc(motion);

    printf("\n\nInitialization is OK, graphical window is launched ...\n\n");
    printf(" s -> start/stop camera rotations\n");
    printf(" o -> on/off auto-pilot engine\n\n");
    printf(" When auto-pilot is turned off, one can use u, h, j, k to control the snake.\n\n");

    glutMainLoop();

    return 0;
#endif

}
